"use strict";
const kcrypto = require('./kcrypto');
const ReqTools = require('kutils/lib/reqtools');
/**
 * 网络请求 - 非单例模式(多用户实例)
 */
module.exports = class {
    constructor(ksession) {
        if (ksession) {
            this.KSession = ksession;
        }
        this.logger = console;
    }
    // 设置日志记录器
    setLogger(logger) {
        this.logger = logger ? logger : console;
    };
    // 异常代码转换
    formatStatus(code) {
        let errtips = '网络错误';
        if (code == 401) {
            errtips = '用户没有被授权,请重新登陆后再尝试';
        } else if (code == 404) {
            errtips = '请求资源不存在';
        } else if (code == 403) {
            errtips = '服务器拒绝请求';
        } else if (code == 500) {
            errtips = '服务器请求异常,请稍后再试';
        } else if (code < 400 && code >= 300) {
            errtips = '[' + code + ']请求已被重定向';
        } else if (code < 500 && code >= 400) {
            errtips = '[' + code + ']无效的请求';
        } else if (code > 500) {
            errtips = '[' + code + ']服务器请求异常,请稍后再试';
        }
        return errtips;
    };

    // 翻译响应结果
    parseAPIRES(res) {
        try {
            if (res.statusCode >= 200 && res.statusCode < 300) {
                let obj = JSON.parse(res.body);
                if (obj.code !== 0) {
                    throw obj.data ? obj.data : '未知错误';
                }
                return obj.data ? obj.data : '';
            } else {
                throw this.formatStatus(res.statusCode);
            }
        } catch (e) {
            this.logger.error(e);
            throw e;
        }
    };

    // 构建带服务器地址的URL
    buildURL(url, params) {
        return ReqTools.buildURL(this.KSession.serverAddr + url, params);
    };

    /**
     * HTTP请求签名函数
     * header签名生成
     */
    buildHeader(accesskey, verb, url, params) {
        // GMT格式时间 (Thu, 17 Mar 2012 18:49:58 GMT)   
        // 注意时间差校对（按 accesskey 中记录的服务器时间(systime) 和本地时间 (localtime) 进行纠偏）
        let offsetseconds = parseInt(accesskey.localtime - accesskey.accesskeytime);
        let time = new Date();
        time.setTime(time.getTime() - offsetseconds * 1000);    //这是服务器当前时间...
        let gmtdate = time.toGMTString();
        let accesskeyid = accesskey.accesskeyid;
        let domain = accesskey.accessdomain;
        //规范化的请求资源
        let signatureStr = ReqTools.buildURL(url, params);
        //签名字符串
        let signature = this.getSignature(verb, gmtdate, signatureStr, accesskey.accesskeysecret);
        let headers = {
            'x-date': gmtdate,
            'x-authorization': domain + " " + accesskeyid + ":" + signature
        }
        return headers;
    };

    // 计算签名
    getSignature(verb, date, signatureStr, secret) {
        let signature = kcrypto.HmacSHA1(verb.toUpperCase() + "\n" + signatureStr + "\n" + date, secret);
        return kcrypto.enc.Base64.stringify(signature);
    };

    // 基础请求
    async request(method, url, opts, notbuild) {
        return new Promise((resolve, reject) => {
            try {
                ReqTools.doRequest(method, notbuild===true?url:this.buildURL(url), opts, (err, body, res) => {
                    try {
                        if (err) {
                            throw err;
                        }
                        resolve(this.parseAPIRES(res));
                    } catch (e) {
                        reject(e);
                    }
                });
            } catch (e) {
                reject(e);
            }
        });
    };

    // API请求
    async requestAPI(method, url, params, opts) {
        return new Promise((resolve, reject) => {
            try {
                if (!opts) {
                    opts = {};
                }
                let session = this.KSession.serverSession;
                if (!session) {
                    throw new Error("not set session, please set it");
                }
                if (typeof session.accesskey.accesskeyid == 'undefined') {
                    throw new Error("accesskeyid is undefined");
                }
                let ak = (session == null || session.accesskey == null || (typeof session.accesskey == 'undefined')) ? null : session.accesskey;
                if (!ak) {
                    throw new Error("accesskey is undefined");
                }
                let reqdatas = {};
                if (params) {
                    for (let key in params) {
                        if (params[key] != undefined && (params[key] + '').length > 0) {
                            reqdatas[key] = params[key];
                        }
                    }
                }
                let kmethod = method == undefined ? "GET" : method;
                let kheader = this.buildHeader(ak, method.toUpperCase(), url, reqdatas);
                opts.headers = Object.assign(Object.assign({}, opts && opts.headers ? opts.headers : {}), kheader);
                opts.form = reqdatas;
		this.request(kmethod, url, opts).then(resolve).catch(reject);
            } catch (e) {
                reject(e);
            }
        });
    }

    // 基础请求 - URL参数传递
    async postURL(url, params, opts) {
        return this.request('POST', ReqTools.buildURL(url, params), opts);
    };

    // API请求
    async postAPI(url, params, opts) {
        return this.requestAPI('POST', url, params, opts);
    }
};