'use strict';
const { EventEmitter } = require('events');
const { ReqTools, FSTools, Utils } = require('kutils');
const KD_EVENT = { PROGRESS: 'progress', ERROR: 'error', ABORT: 'abort', END: 'end' };
const KD_OPTS = { RETRAY: 'retray', COVER: 'cover', STOP: 'stop' };
/**
 * 下载单个文件
    let kdh = new KDownloadHandler('M:/download-test.exe', { filepath: '@900000000xtenantdoc/docs/download-test.exe' });
    kdh.on('error', (err) => { });
    kdh.on('progress', (ps) => { });
    kdh.on('end', () => { });
    kdh.start();
 */
module.exports = class extends EventEmitter {
    // 
    constructor(localpath, opts, kspace) {
        super();
        if (!localpath) {
            throw new Error('localpath is empty');
        }
        this._localpath = localpath; this._translatedsize = 0; this._timeStart = -1;
        this._opts = Utils.extendAttr({ filepath: '', expiredsecond: '', downfilename: '', hopesrvs: '', downmimetype: '', datahash: '', rangestart: -1, rangestop: -1, isfile: true }, opts);
        if (!this._opts.filepath && !this._opts.datahash) {
            throw new Error('filepath is empty');
        }
        this.KSpace = kspace ? kspace : require('kapis/lib/kspace');
        this.request = null; this.writeStream = null;
    };
    // 启动
    start() {
        try {
            // 基础校验
            let exists = FSTools.isExistsSync(this._localpath);
            if (exists) {
                let isfile = FSTools.isFileSync(this._localpath);
                if (isfile) {
                    this.emit(KD_EVENT.ERROR, { msg: '文件已存在[' + this._localpath + ']', stage: 'validation', opts: [KD_OPTS.RETRAY, KD_OPTS.COVER] }); return;
                }else{
                    this.emit(KD_EVENT.END, this._localpath, 0); 
                    return;
                }
            } else {
                if(this._opts.isfile===false){
                    this._doMkDirs();
                    return;
                }else{
                    FSTools.mkDIRsSync(FSTools.getParent(this._localpath));
                }
            }
            // 
            this._askDownload((datas) => {
                try {
                    this._writeLocal(datas);
                } catch (e) {
                    this.emit(KD_EVENT.ERROR, { error: err, msg: '下载失败[' + err.toString() + ']', stage: 'request', opts: [KD_OPTS.RETRAY] });
                }
            });
        } catch (err) {
            this.emit(KD_EVENT.ERROR, { error: err, msg: '下载失败[' + err.toString() + ']', stage: 'validation', opts: [KD_OPTS.RETRAY] });
        }
    };
    async _doMkDirs() {
        try {
            if (!FSTools.isExistsSync(this._localpath)) {
                FSTools.mkDIRsSync(this._localpath);
            }
            this.emit(KD_EVENT.END, this._localpath, 0);
        } catch (err) {
            this.emit(KD_EVENT.ERROR, { error: err, msg: '下载失败[' + err.toString() + ']', stage: 'mkdirs', opts: [KD_OPTS.RETRAY] });
        }
    }
    // 
    _getCachePath() {
        return this._localpath + '.downloading';
    };
    // 获取下载信息
    _askDownload(cb) {
        // 获取下载url
        // datahash: 'fa1f5ab72757ade04b5bce4d6dc7654c'
        // datasize: 165731
        // datasrv: 2001
        // datatype: 'pdf'
        // downname: 'eFapiao_20201121R3895927685_94056425.pdf'
        // downtoken: 'down002001qcksmreh20122016445506kabalafa1f5ab72757ade04b5bce4d6dc7654cx165731x202011280947280004'
        // expiredtime: 20201220164455
        // url: 'http://192.168.2.72:5775/kabala/2001/token/down002001qcksmreh20122016445506kabalafa1f5ab72757ade04b5bce4d6dc7654cx165731x202011280947280004'
        this.KSpace.askdownload(...Object.values(this._opts)).then(cb).catch((err) => {
            this.emit(KD_EVENT.ERROR, { error: err, msg: '下载失败[' + err.toString() + ']', stage: 'askdownload', opts: [KD_OPTS.RETRAY] });
        });
    };
    // 数据流传送
    _writeLocal(datas) {
        let cachePath = this._getCachePath();
        // 断点头信息.
        let requestOpts = {}; let writeOpts = {};
        if (this._opts.rangestart > 0) {
            this._translatedsize = this._opts.rangestart;
            requestOpts.headers = { 'Range': 'bytes=' + this._opts.rangestart + '-' + (this._opts.rangestop > 0 ? this._opts.rangestop : '') };
            writeOpts.start = this._opts.rangestart;
            writeOpts.flags = 'r+';
        } else {
            if (FSTools.isFileSync(cachePath)) {
                FSTools.removeSync(cachePath);
            }
        }
        // 
        this._timeStart = -1;
        this.request = ReqTools.doRequest('GET', datas.url, requestOpts);
        this.writeStream = FSTools.writeStream(cachePath, writeOpts);
        this.request.on('error', (err) => {
            this.writeStream.removeAllListeners('close');
            this.writeStream.close();
            this.writeStream.end();
            this.emit(KD_EVENT.ERROR, { error: err, msg: '下载失败[' + err.toString() + ']', stage: 'request', opts: [KD_OPTS.RETRAY] });
        });
        // 保存流到缓存文件.
        this.request.on('response', (response) => {
            this._timeStart = new Date().getTime();
            this.writeStream.on('error', (err) => {
                this.request.removeAllListeners('end');
                this.writeStream.removeAllListeners('close');
                this.emit(KD_EVENT.ERROR, { error: err, msg: '下载失败, 文件写入问题[' + err.toString() + ']', stage: 'writeSteam', opts: [KD_OPTS.RETRAY] });
            });
            this.writeStream.on('drain', () => {
                response.resume();
            });
            this.writeStream.on('close', () => {
                try {
                    if (!FSTools.isFileSync(cachePath)) {
                        this.emit(KD_EVENT.ERROR, { msg: '缓存文件不存在, 文件是否被第三方程序删除?[' + cachePath + ']', stage: 'cacheRefresh', opts: [KD_OPTS.RETRAY] }); return;
                    }
                    if (FSTools.isFileSync(this._localpath)) {
                        this.emit(KD_EVENT.ERROR, { msg: '文件已存在[' + this._localpath + ']', stage: 'cacheRefresh', opts: [KD_OPTS.RETRAY, KD_OPTS.COVER] }); return;
                    }
                    FSTools.moveAsync(cachePath, this._localpath, (err) => {
                        if (err) {
                            this.emit(KD_EVENT.ERROR, { msg: '下载失败[' + err.toString() + ']', stage: 'cacheRefresh', opts: [KD_OPTS.RETRAY] }); return;
                        }
                        this.emit(KD_EVENT.END, this._localpath, datas.datasize);
                    });
                } catch (err) {
                    this.emit(KD_EVENT.ERROR, { error: err, msg: '下载失败[' + err.toString() + ']', stage: 'cacheRefresh', opts: [KD_OPTS.RETRAY] });
                }
            });
            response.on('error', (err) => {
                this.writeStream.removeAllListeners('close');
                this.writeStream.close();
                this.writeStream.end();
                this.emit(KD_EVENT.ERROR, { error: err, msg: '下载失败[' + err.toString() + ']', stage: 'response', opts: [KD_OPTS.RETRAY] });
            });
            response.on('end', () => {
                this.writeStream.end();
            });
            response.on('abort', () => {
                this.writeStream.removeAllListeners('close');
                this.writeStream.close();
                this.writeStream.end();
                this.emit(KD_EVENT.ABORT, { msg: '下载中止', stage: '' });
            });
            response.on('data', (chunk) => {
                if (chunk && chunk.length > 0) {
                    if (!this.writeStream.write(chunk)) {
                        response.pause();
                    }
                    this._translatedsize += chunk.length;
                    let spentTime = new Date().getTime() - this._timeStart;
                    let ps = Utils.calcPercentage(this._translatedsize, datas.datasize);
                    let speed = this._translatedsize / spentTime; // byte / ms 
                    this.emit(KD_EVENT.PROGRESS, { translatedSize: this._translatedsize, dataSize: datas.datasize, spentTime: spentTime, progress: ps, speed: speed });
                }
            });
        });
    };

    // 停止
    abort() {
        if (this.request) {
            this.request.abort();
        }
    };
};