'use strict';
const fileSys = require('fs');
const crypto = require('crypto');
/**
 * 本地文件操作类
 * fstools.js
 **/
module.exports = class {
	/**
	 * windows 设备需要转换路径为 D:/a\b/c ==> D:\a\b\c
	 * Unix 设备只要规范就行 /a/b/c
	 * isunix 不传, 则自动使用设备平台
	**/
	static parsePath(path, isunix) {
		if (!path) {
			return path;
		}
		let temp = path; // .parsePath();
		if (false === isunix || (undefined == isunix && process.platform == 'win32')) {
			let isnet = temp.indexOf(":")!=1;  //网络路径
			if (temp.indexOf('/') > -1) {
				temp = temp.replace(new RegExp('/', 'gm'), '\\');
			}
			if (temp.indexOf('\\\\') > -1) {
				temp = temp.replace(new RegExp('\\\\\\\\', 'gm'), '\\');
			}
			let indexof = temp.indexOf('\\');
			if (indexof == -1) {
				temp += '\\';
			} else {
				let subindex = temp.lastIndexOf('\\');
				if (subindex > indexof && subindex == temp.length - 1) {
					temp = temp.substring(0, subindex);
				}
			}
			if(isnet){
				temp = "\\"+temp;
			}
		} else {
			if (temp.indexOf('\\') > -1) {
				temp = temp.replace(new RegExp('\\\\', 'gm'), '/');
			}
			if (temp.indexOf('//') > -1) {
				temp = temp.replace(new RegExp('//', 'gm'), '/');
			}
			if ('/' != temp) {
				let subindex = temp.lastIndexOf('/');
				if (subindex > -1 && subindex == temp.length - 1) {
					temp = temp.substring(0, subindex);
				}
			}
		}
		return temp;
	};
	// 获取父级
	static getParent(path, isunix) {
		if (!path) {
			return path;
		}
		let temp = this.parsePath(path, isunix);
		if (false === isunix || (undefined == isunix && process.platform == 'win32')) {
			if (temp.indexOf('\\') > -1) {
				temp = temp.substring(0, temp.lastIndexOf('\\'));
			}
		} else {
			if (temp.indexOf('/') > -1) {
				temp = temp.substring(0, temp.lastIndexOf('/'));
			}
		}
		return temp;
	};
	// 获取名字
	static getName(path, hasSuffixed) {
		let temp = this.parsePath(path, true);
		temp = temp.substring(temp.lastIndexOf('/') + 1);
		if (hasSuffixed == false && temp.lastIndexOf('.') != -1) {
			temp = temp.substring(0, temp.lastIndexOf('.'));
		}
		return temp;
	};
	// 构建路径
	static buildPath(base, addstr, isunix) {
		let b = undefined != base ? base.trim() : '';
		let a = undefined != addstr ? addstr.trim() : '';
		let s = isunix ? '/' : (undefined == isunix ? process.platform == 'win32' ? '\\' : '/' : '\\');
		return this.parsePath(b + s + a, isunix);
	};

	// 是否存在, 文件或者文件夹
	static isExistsSync(filePath) {
		return fileSys.existsSync(filePath);
	};

	// 是否是文件
	static isFileSync(filePath) {
		let isFile = false;
		if (this.isExistsSync(filePath)) {
			isFile = fileSys.statSync(filePath).isFile();
		}
		return isFile;
	};

	// 是否是文件夹
	static isDIRSync(dirPath) {
		let isDir = false;
		try {
			if (this.isExistsSync(dirPath)) {
				isDir = fileSys.statSync(dirPath).isDirectory();
			}
		} catch (e) { }
		return isDir;
	};

	// 获取文件信息
	static infoSync(filePath) {
		return fileSys.statSync(filePath)
	};

	// 获取文件大小
	static fileSizeSync(filePath) {
		filePath = this.parsePath(filePath);
		if (this.isExistsSync(filePath)) {
			return fileSys.statSync(filePath).size;
		} else {
			return 0;
		}
	};

	// 重命名
	static renameSync(filePath, newName) {
		try {
			let tpath = this.parsePath(filePath,);
			if (!this.isExistsSync(tpath)) { return false; };
			fileSys.renameSync(tpath, this.buildPath(this.getParent(filePath), newName));
		} catch (e) { return false; }
		return true;
	};

	//创建多层文件夹
	static mkDIRsSync(dir, mode) {
		dir = this.parsePath(dir, true);
		if (!this.isExistsSync(this.parsePath(dir))) {
			let paths = dir.split('/');
			let tempPath = '';
			for (let i = 0; i < paths.length; i++) {
				if (i == 0) {
					tempPath = paths[i] + '/';
				} else {
					tempPath = tempPath + '/' + paths[i];
				}
				tempPath = this.parsePath('/' == tempPath ? tempPath : tempPath);
				if (this.isExistsSync(tempPath)) {
					let states = fileSys.statSync(tempPath);
					if (states.isFile()) {
						fileSys.mkdirSync(tempPath, mode);
						fileSys.chmodSync(tempPath, 0o775);
					}
				} else {
					fileSys.mkdirSync(tempPath, mode);
					fileSys.chmodSync(tempPath, 0o775);
				}
			}
		}
	};

	// 获取文件列表( 源路径, 包含子文件夹, 包含文件夹路径, 包含文件 )
	static listDIRSync(dir, hasChild, hasDIR, hasFile) {
		hasChild = undefined == hasChild ? false : hasChild;
		hasDIR = undefined == hasDIR ? true : hasDIR;
		hasFile = undefined == hasFile ? true : hasFile;
		let fileList = new Array();
		dir = this.parsePath(dir);
		// 
		let loopFiles = (dir_loop) => {
			let files = fileSys.readdirSync(dir_loop);
			files.forEach((file) => {
				try {
					let states = fileSys.statSync(this.parsePath(dir_loop + '/' + file));
					if (states != null) {
						let tempPath = this.parsePath(dir_loop + '/' + file);
						if (states.isDirectory()) {
							if (hasDIR) {
								fileList.push(tempPath);
							}
							if (hasChild) {
								loopFiles(tempPath);
							}
						} else if (hasFile) {
							fileList.push(tempPath);
						}
					}
				} catch (e) { }
			});
		};
		if (this.isExistsSync(dir)) {
			loopFiles(dir);
		}
		return fileList;
	};

	// 获取文件列表( 源路径, {包含子文件夹, 包含文件夹路径, 包含文件}, callback )
	static listDIRAsync(filePath, option, callback) {
		let temppath = this.parsePath(filePath);
		let hasChild = option.hasChild == undefined ? false : option.hasChild;
		let hasDIR = option.hasDIR == undefined ? true : option.hasDIR;
		let hasFile = option.hasFile == undefined ? true : option.hasFile;

		let result = new Array();
		let loopFiles = (filePath_loop, nextCallback) => {
			fileSys.readdir(filePath_loop, (err, list) => {
				if (err) { if (callback) { callback(err); }; return; }
				let nextStat = (i) => {
					if (list.length == i) {
						if (nextCallback) { nextCallback(); };
						return; // 当前文件夹已经循环完毕, 返回
					}
					if (!list[i] || list[i].length == '') {
						nextStat(++i); return;
					}
					let folderDir = this.parsePath(filePath_loop + '/' + list[i]); // 子路径
					fileSys.stat(folderDir, (stat_err, states) => {
						if (stat_err) { if (callback) { callback(stat_err); }; return; }
						if (states) {
							if (states.isDirectory()) {
								if (hasDIR) {
									result.push(folderDir);
								}
								if (hasChild) {
									loopFiles(folderDir, () => {
										nextStat(++i);
									});
								} else {
									nextStat(++i);
								}
							} else {
								if (hasFile) {
									result.push(folderDir);
								}
								nextStat(++i); // 如果是文件则循环当前文件夹其他项目
							}
						} else {
							nextStat(++i); // 当前文件夹项目无效, 继续当前文件夹其他项目
						}
					});
				};
				if (!list || list.length == 0) {
					if (nextCallback) { nextCallback(); };
				} else {
					nextStat(0);
				}
			});
		};
		// 
		if (this.isExistsSync(temppath)) {
			loopFiles(temppath, () => {
				if (callback) { callback(false, result); }
			});
		} else {
			if (callback) { callback(new Error(filePath + ' is not exists')); }
		}
	};

	// 获取文件数据流
	static readStreamSync(filePath, option) {
		return fileSys.createReadStream(filePath, option);
	};

	// 获取文件数据流
	static writeStream(filePath, option) {
		return fileSys.createWriteStream(filePath, option);
	};

	// 获取单片数据流 option: { filePath, partNumber, partSize, startSize }
	static readPartStreamSync(option) {
		let partSize = option.partSize ? option.partSize : 1024 * 1024 * 5;
		let startSize = option.startSize ? option.startSize : '';
		let filePath = option.filePath ? this.parsePath(option.filePath) : '';
		let partNumber = option.partNumber ? option.partNumber : 1;

		if (this.isFileSync(filePath)) {
			let fs_stata = fileSys.statSync(filePath);
			let file_size = fs_stata.size;
			// 0 ~ 4 ==> lengh=5 5 ~ 9   10 ~ 14 15 ~ 19
			let file_start = (partNumber <= 1 ? 0 : partSize * (partNumber - 1));
			let file_end = file_start + partSize - 1;

			// 验证范围
			if (file_start >= file_size && file_size != 0) {
				throw new Error('beyond the scope, Path: ' + filePath);
			}
			// 修正超出范围
			if (file_end >= file_size) {
				file_end = file_size - 1;
			}
			// 文件分片信息
			let readOption = {
				fileSize: file_size,
				bufferSize: partSize,
				contentSize: file_end - file_start + 1,
				start: file_start,
				end: file_end
			};
			// 修正内容长度
			if (file_size == 0) {
				readOption.contentSize = 0;
			}
			// 修正指定开始位置
			if ('' != startSize && startSize > file_start) {
				readOption.start = startSize;
			}
			// 
			let readSteam = this.readStreamSync(filePath, file_size == 0 ? {} : readOption);
			readSteam.option = readOption;
			return readSteam;	
		}
	};

	// 获取文件不重复路径,用于文件重复是
	static getSavePath(savePath, isDirectory) {
		savePath = this.parsePath(savePath);
		if (!this.isExistsSync(savePath)) {
			return savePath;
		}
		let pathIndex = 1;
		if (isDirectory == true) { // 文件夹路径
			while (true) {
				let tempPath_2 = savePath + '(' + pathIndex + ')';
				if (!this.isExistsSync(tempPath_2)) {
					return tempPath_2;
				}
				pathIndex++;
			}
		} else {  // 文件路径
			let filesuffixed = savePath.getSuffixed();
			let tempPath_1 = this.buildPath(this.getParent(savePath), this.getName(savePath, false));
			while (true) {
				let tempPath_2 = tempPath_1 + '(' + pathIndex + ')' + filesuffixed;
				if (!this.isExistsSync(tempPath_2)) {
					return tempPath_2;
				}
				pathIndex++;
			}
		}
	};

	// 复制单个文件操作
	static copyFileAsync(src, dst, callback) {
		src = this.parsePath(src);
		if (this.isFileSync(src)) {
			let readable = fileSys.createReadStream(src);
			let writable = fileSys.createWriteStream(this.parsePath(dst));
			writable.on('close', callback);
			readable.pipe(writable);
			return { readable: readable, writable: writable };
		} else {
			callback('Can\'t find the source file.');
		}
	};

	// 复制文件夹
	static copyDIRAsync(src, dst, callback) {
		if (callback) {
			if (src == dst || src == '' || '' == dst) {
				callback('Parameter error.');
			}
		} else { return; }
		src = this.parsePath(src);
		dst = this.parsePath(dst);
		let fileListSrc = this.listDIRSync(src, true, true, true);
		let fileListDst = new Array();
		for (let i = 0; i < fileListSrc.length; i++) {
			fileListDst.push(fileListSrc[i].replace(src, dst));
		}
		if (fileListDst && fileListDst.length > 0) {
			// do function
			let doCopy = (index) => {
				if (fileListSrc.length == index) { // end
					callback(); return;
				}
				let dirOrFilePathSrc = fileListSrc[index];
				let dirOrFilePathDst = fileListDst[index];
				let dirOrFilePathDst_Parent = this.getParent(dirOrFilePathDst);
				if (!this.isDIRSync(dirOrFilePathDst_Parent)) {
					if (!this.mkDIRsSync(dirOrFilePathDst_Parent)) {
						callback('Failed to create the folder.'); return;
					}
				};
				if (this.isFileSync(dirOrFilePathSrc)) {
					this.copyFileAsync(dirOrFilePathSrc, dirOrFilePathDst, () => {
						doCopy(++index);
					});
				} else {
					if (this.mkDIRsSync(dirOrFilePathSrc)) {
						doCopy(++index);
					} else {
						callback('Failed to create the folder.');
					}
				}
			};
			doCopy(0); // do
		} else {
			callback('Can\'t find the source file or dir.');
		}
	};

	/**
	 * 移动文件
	 * 同步移动不支持支持跨分区移动
	 **/
	static moveSync(oldPath, newPath) {
		oldPath = this.parsePath(oldPath);
		newPath = this.parsePath(newPath);
		try {
			if (!this.isExistsSync(oldPath)) { return false; };
			if (this.isExistsSync(newPath)) { return false; };
			if (!this.isExistsSync(this.getParent(newPath))) {
				if (!this.mkDIRsSync(this.getParent(newPath))) {
					return false;
				};
			};
			fileSys.renameSync(oldPath, newPath);
		} catch (e) {
			/*if( e.message ){
				// 跨设备
				if(e.message.toString( ).indexOf('cross-device') != -1 ){
					fileSys.createReadStream( oldPath ).pipe( fileSys.createWriteStream( newPath ) );
					fileSys.unlinkSync( oldPath );
					return true;
				}
			}*/
			throw e;
		}
		return true;
	};

	// 移动文件
	/**
	 * 移动文件
	 * 支持跨分区移动
	 **/
	static moveAsync(oldPath, newPath, callback) {
		let _CallBack = (err) => {
			if (callback) { callback(err); };
		}
		try {
			oldPath = this.parsePath(oldPath);
			newPath = this.parsePath(newPath);
			if (!this.isExistsSync(oldPath)) {
				_CallBack(new Error('src path is not exists')); return;
			};
			if (this.isExistsSync(newPath)) {
				_CallBack(new Error('dest path is exists')); return;
			};
			if (!this.isExistsSync(this.getParent(newPath))) {
				this.mkDIRsSync(this.getParent(newPath));
			};
			fileSys.rename(oldPath, newPath, (err) => {
				if (err && err.code.toLowerCase() == 'exdev') {
					fileSys.createReadStream(oldPath).pipe(fileSys.createWriteStream(newPath)).on('unpipe', () => {
						fileSys.unlink(oldPath, _CallBack);
					});
				} else {
					_CallBack(err);
				}
			});
		} catch (err_Catch) {
			_CallBack(err_Catch);
		}
	};

	// 删除文件/夹 可选是否删除当前目录或清除以下子目录
	static removeSync(removePath, removeRoot) {
		let _ = this;
		(function doDelete(filePath, isRemoveRoot, isRoot) {
			let files = [];
			if (_.isExistsSync(filePath)) {
				if (fileSys.statSync(filePath).isFile()) {
					fileSys.unlinkSync(filePath);
				} else {
					files = fileSys.readdirSync(filePath);
					files.forEach((file) => {
						let curPath = _.parsePath(_.buildPath(filePath, file));
						if (fileSys.statSync(curPath).isDirectory()) { // recurse
							doDelete(curPath, true, false);
						} else { // delete file
							fileSys.unlinkSync(curPath);
						}
					});
					if (!isRoot) {
						fileSys.rmdirSync(filePath);
					} else if (isRoot && isRemoveRoot) {
						fileSys.rmdirSync(filePath);
					}
				}
			}
		})(this.parsePath(removePath), removeRoot);
	};

	// 获取大文件的MD5 返回ReadStream,结果通过回掉返回
	static getMD5Async(filePath, callback) {
		filePath = this.parsePath(filePath);
		let md5sum = crypto.createHash('md5');
		if (this.isFileSync(filePath)) {
			let stream = fileSys.createReadStream(filePath);
			stream.on('data', (chunk) => {
				md5sum.update(chunk);
			});
			stream.on('end', () => {
				callback(md5sum.digest('hex').toUpperCase());
			});
			return stream
		}
	};

	// 覆盖写入文本文件
	static writeFileSync(savePath, text, option) {
		try {
			savePath = this.parsePath(savePath);
			if (!this.isExistsSync(savePath)) {
				this.mkDIRsSync(this.getParent(savePath));
			}
			fileSys.writeFileSync(savePath, text, option ? option : {});
		} catch (e) { return false; }
		return true;
	};

	// 覆盖写入文本文件
	static writeFileAsync(savePath, text, option, callback) {
		savePath = this.parsePath(savePath);
		if (!this.isExistsSync(savePath)) {
			this.mkDIRsSync(this.getParent(savePath));
		}
		fileSys.writeFile(savePath, text, option ? option : {}, callback);
	};

	// 追加写入文本文件
	static appendFileAsync(savePath, text, option, callback) {
		savePath = this.parsePath(savePath);
		if (!this.isExistsSync(savePath)) {
			this.mkDIRsSync(this.getParent(savePath));
		}
		fileSys.appendFile(savePath, text, option ? option : {}, callback);
	};

	// 追加写入文本文件
	static appendFileSync(savePath, text, option) {
		try {
			savePath = this.parsePath(savePath);
			if (!this.isExistsSync(savePath)) {
				this.mkDIRsSync(this.getParent(savePath));
			}
			fileSys.appendFileSync(savePath, text, option ? option : {});
		} catch (e) { return false; }
		return true;
	};

	// 读取文本文件
	static readFileSync(readPath, option) {
		readPath = this.parsePath(readPath);
		if (this.isExistsSync(readPath)) {
			return fileSys.readFileSync(readPath, option ? option : {});
		} else {
			return '';
		}
	};

	// 读取文本文件
	static readFileAsync(readPath, option, callback) {
		readPath = this.parsePath(readPath);
		fileSys.readFile(readPath, option ? option : {}, callback);
	};

	// 获取程序配置信息
	static getJson(configPath, item, isBase64) {
		try {
			configPath = this.parsePath(configPath)
			if (!this.isExistsSync(configPath)) {
				return '';
			}
			let ret = fileSys.readFileSync(configPath, 'utf-8');
			if (ret.length == 0) {
				return '';
			}
			let appConfig = JSON.parse(isBase64 != false ? Buffer.from(ret, 'base64').toString() : ret);
			if (item) {
				return appConfig[item] == undefined ? '' : appConfig[item];
			} else {
				return appConfig;
			}
		} catch (e) {
			return '';
		}
	};

	// 设置一个配置信息 保存|更新键值对|重置内容
	static setJson(configPath, item, value, op, isBase64) {
		try {
			let json_ = {};
			configPath = this.parsePath(configPath);
			if (this.isExistsSync(configPath)) {
				let ret = fileSys.readFileSync(configPath, 'utf-8');
				if (ret.length != 0) {
					if (isBase64 != false) {
						json_ = eval('(' + Buffer.from(ret, 'base64').toString() + ')');
					} else {
						json_ = eval('(' + ret + ')');
					}
				}
			}
			if (op == 'remove' && item != '') {
				delete json_[item];
			} else if (op == 'clear') {
				json_ = {};
			} else if ((!item || item == '') && value != '' && typeof (value) == 'object') {
				json_ = value;
			} else {
				json_[item] = value;
			}
			let body = JSON.stringify(json_);
			if (isBase64 != false) { body = Buffer.from(body).toString('base64'); }
			let parentPath = this.getParent(configPath);
			if (!this.isExistsSync(parentPath)) {
				if (!this.mkDIRsSync(parentPath)) {
					throw 'setAppConfig mkdirsSync err';
				}
			}
			fileSys.writeFileSync(configPath, body);
			body = null; json_ = null;
		} catch (Err_Catch) {
			throw Err_Catch;
		}
		return true;
	};

	static isFreed(filePath){
		try{
			if(!this.isFileSync(filePath)){
				console.log('file not exits')
				return false;
			}
			let oldParentPath = this.getParent(filePath);
			let oldParentDir = oldParentPath.getName();
			let newParentDir = oldParentDir+'_t';
			let newParentPath = this.buildPath(this.getParent(oldParentPath),newParentDir);
			if(!this.renameSync(oldParentPath, newParentDir)){
				console.log('dir rename fail')
				return false;
			}
			this.renameSync(newParentPath, oldParentDir)
			return true;
		}catch(e){
			console.log('exception')
			return false;
		}

	}
};

// 测试方法
let FSToolsTest = function () {
	let os = require('os');
	let FSTools = require(__filename);
	let tempdir = os.tmpdir();
	console.log('System temp dir', tempdir);
	// uninx 路径转换
	tempdir = FSTools.parsePath(tempdir, true);
	console.log('Parse Unix path: ', tempdir);
	tempdir = FSTools.buildPath(FSTools.parsePath(tempdir, true), '/fstools\\testdir/');
	console.log('Test run dir: ', tempdir);
	let mkdir = FSTools.mkDIRsSync(tempdir)
	if (mkdir) {
		console.log('mkDIRsSync ', tempdir);
	}
	for (let i = 0; i < 10; i++) {
		FSTools.writeFileSync(FSTools.buildPath(tempdir, i + '.text'), 'testfile-' + i);
	}
	// let dirList = FSTools.listDIRSync(FSTools.getParent(tempdir), true, false);
	FSTools.listDIRAsync(FSTools.getParent(tempdir), { hasChild: true, hasDIR: false }, (err, list) => {
		if (err) {
			console.error(err); return;
		}
		// json 测试
		let jsonPath = FSTools.buildPath(tempdir, 'jsontest.json');
		FSTools.setJson(jsonPath, '', { dirlist: list }, '', true);
		console.log('Write json file: ', jsonPath);
		let cftvalue = FSTools.getJson(jsonPath, 'dirlist', true);
		console.log('Read json key(dirList): ', cftvalue);
		let testdir1 = FSTools.buildPath(FSTools.getParent(tempdir), 'testdir1');
		FSTools.copyDIRAsync(tempdir, testdir1, (err) => {
			if (err) {
				console.error(err);
			} else {
				console.log('copyDIRAsync', testdir1)
			}
			// 删除测试
			let removed = FSTools.removeSync(tempdir, true);
			if (removed) {
				console.log('DIR remov: ', tempdir)
			}
			// 
			let dirs = FSTools.listDIRSync(FSTools.getParent(tempdir), true);
			console.log('removed dir: ', dirs)
			FSTools.removeSync(FSTools.getParent(tempdir), true)
		});
	});




};