"use strict";
const { Utils, FrameTool, FSTools } = require('kutils');
const KFrameAbstract = require('ksdk/lib/kframeabstract');
/**
 * 传输抽象类
 */
module.exports = class {
    constructor(type) {
        this._errFrame = null;     // 
        this._queue = new Map();   // 任务队列
        this._queueoffset = 0;     // 当前任务队列下标
        this._transferobj = null;  // 传输实例
        this.isabort = false; this.logger = console;
        this._ps = {
            title: '正在传输', // 标题 -
            src: '',        // 原位置
            dest: '',       // 目标位置
            translated: 0,  // 单个已传输大小
            dataSize: 0,    // 单个文件大小
            speed: 0,       // 单个网速
            allcount: 0,    // 共计大小 -
            allsucceed: 0,  // 总计成功 -
            allbreak: 0,    // 总计跳过
            allSize: 0,     // 全部大小 -
            alltranslated: 0,   // 一共已传输大小 -
            transferend: false,  // 传输结束
            buildQueueEnd: false,   // 异步队列是否完成
        };
        this.lastopts = {
            opt: '',       // 最后执行的操作
            always: false, // 记住
        };
        if (!type) {
            throw new Error('type is empty');
        }
        this._frame = new TransferFrame(type);
        // 
        this._frame.win.on('closed', (e) => { this._frame = null; });
        // 注册控制事件 { opt: 'retry, stop, break', always: false }
        this._frame.doRegisterHandle('control', (e, opt, always) => {
            let opttemp = opt;
            if (this.lastopts.always && 'stop' != opttemp) {
                opttemp = this.lastopts.opt;
            } else {
                this.lastopts.opt = opt;
                this.lastopts.always = ('break' == opttemp || 'cover' == opttemp) && always;
            }
            if (this._errFrame) {
                try { this._errFrame.win.hide(); } catch (e) { this.logger.error(e); }
            }
            this.onControl(opttemp);
        });
    };
    // 执行
    async doAction() {
        try {
            let win = await this.showPage();
            win.show(); win.focus(); win.setAlwaysOnTop(true);
            // win.webContents.openDevTools()
            await this.doBuildQueue((item) => {
                this._addQueue(item);
            });
            this._onBuildQueueEnd();
        } catch (err) {
            this.logger.error(err);
            this.stop(); throw err;
        }
    };
    // 停止
    stop() {
        this.isabort = true;
        this._queue.clear();
        if (null != this._transferobj) {
            try {
                this._transferobj.abort();
            } catch (e) { this.logger.error(e); }
            this._transferobj = null;
        }
        if (null != this._frame) {
            try {
                this._frame.win.destroy();
            } catch (e) { this.logger.error(e); }
            this._frame = null;
        }
        if (null != this._errFrame) {
            try {
                this._errFrame.win.destroy();
            } catch (e) { this.logger.error(e); }
            this._errFrame = null;
        }
    };
    // 设置标题
    setTitle(title) {
        if (title) {
            this._ps.title = title;
        }
    };
    // 
    // 添加传输队列 item={src, dst, opts}
    _addQueue(item) {
        if (undefined == item.datasize) {
            throw 'item.datasize cannot undefined';
        }
        // 
        this._ps.allcount++;
        this._ps.allSize += parseInt(item.datasize);
        this._queue.set(this._queue.size, item);
        this.onProcess();
        this._triggerTransfer();
    };
    // 异步队列结束
    _onBuildQueueEnd() {
        this._ps.buildQueueEnd = true;
        this._triggerTransfer();
    };
    // 触发传输动作
    async _triggerTransfer() {
        if (this.isabort) {
            // throw 'object is aborted';
            return;
        }
        if (null == this._transferobj && this._queue.size > 0) {
            // 传完了
            if (this._ps.buildQueueEnd) {
                if (this._queue.size <= this._queueoffset) {
                    this._ps.transferend = true;
                    this.onProcess();
                    this.onTransferEnd({
                        allcount: this._ps.allcount,
                        allsucceed: this._ps.allsucceed,
                        allbreak: this._ps.allbreak,
                    });
                    return;
                }
            }
            // 新建传输
            try {
                if(this._queue.size <=this._queueoffset ){
                    return;
                }
                let item = this._queue.get(this._queueoffset);
                this.doBuildItem(item);
                this._queue.set(this._queueoffset, item);
                this._transferobj = this.doGetTransferObject(item);
                this._transferobj.on('error', (err) => {
                    this.onError(err);
                });
                this._transferobj.on('progress', (ps) => {
                    this.onProcess(item, ps);
                });
                this._transferobj.on('end', (_) => {
                    this._ps.alltranslated += parseInt(item.datasize);
                    this._ps.allsucceed++;
                    this._queueoffset++;
                    this._transferobj = null;
                    this.onProcess(item);
                    this._triggerTransfer();
                });
                this._transferobj.start();
            } catch (err) {
                if (null != this._transferobj) {
                    this._transferobj.abort();
                }
                if (!err.error && !err.msg) {
                    err = { error: err, msg: err.toString(), stage: 'unknown', opts: ['retray'] }
                }
                this.onError(err);
                this.logger.error(err);
            }
        }
    };
    // 进度刷新
    onProcess(item, ps) {
        let currentps = {
            title: this._ps.title,
            src: '', dest: '',
            translated: 0, dataSize: 0, speed: 0,
            allcount: this._ps.allcount,
            allsucceed: this._ps.allsucceed,
            allbreak: this._ps.allbreak,
            allSize: this._ps.allSize,
            alltranslated: this._ps.alltranslated,
            transferend: this._ps.transferend,
        };

        if (item) {
            currentps.src = item.showsrc ? item.showsrc : item.src;
            currentps.dest = item.showdest ? item.showdest : item.dest;
        }
        if (ps) {
            // ps = { translatedSize: this._translatedsize, dataSize: datas.datasize, spentTime: spentTime, progress: ps, speed: speed }
            if (ps.stage == 'calcmd5') {
                currentps.calcprocess = ps.progress ? '正在计算:' + ps.progress : '正在计算';
            } else {
                currentps.translated = ps.translatedSize;
                currentps.dataSize = ps.dataSize;
                currentps.speed = ps.speed ? ps.speed : 0;
                currentps.alltranslated += currentps.translated;
            }
        }
        if (this._frame && this._frame.win) {
            this._frame.win.webContents.send('t-ps', currentps);
        }
    };
    // opsts
    onControl(opttemp) {
        if (this.isabort) {
            opttemp = 'stop';
        }
        this._transferobj = null;
        if ('stop' == opttemp) {
            this.stop();
        } else if ('retry' == opttemp) {
            this._triggerTransfer();
        } else if ('break' == opttemp) {
            this._queueoffset++;
            this._ps.allbreak++;
            let item = this._queue.get(this._queueoffset);
            this.onProcess(item);
            this._triggerTransfer();
        } else if ('cover' == opttemp) {
            let item = this._queue.get(this._queueoffset);
            if (item.dest) {
                try {
                    FSTools.removeSync(item.dest);
                } catch (e) {
                    this.logger.error(e);
                }
            }
            this._triggerTransfer();
        }
        // this.onProcess();
    };
    // 有错误
    onError(err) {
        // 
        if (this.isabort) { return; }
        // 记住选项
        if (this.lastopts.always) {
            setTimeout(() => { this.onControl(this.lastopts.opt); }); return;
        }
        if (this._frame) {
            // this._frame.win.webContents.send('t-err', err);
            if (null == this._errFrame) {
                this._errFrame = new TransferErrFrame(this._frame.id);
                this._errFrame.doLoadFile('./page/page-kcomm/transfererror.html');
                // this._errFrame.win.webContents.openDevTools()
                this._errFrame.win.on('ready-to-show', () => {
                    this._errFrame.win.show();
                    this._errFrame.win.focus();
                    this._errFrame.win.setAlwaysOnTop(true);
                    this._errFrame.win.webContents.send('t-err', err);
                });
                this._errFrame.win.on('closed', () => {
                    this._errFrame = null;
                });
            } else {
                this._errFrame.win.webContents.send('t-err', err);
                this._errFrame.win.show();
                this._errFrame.win.focus();
                this._errFrame.win.setAlwaysOnTop(true);
            }
        }
    };
    // 启动页面 - override
    async showPage() {
        return this._frame.doLoadFile('./page/page-kcomm/transfer.html');
    };
    // 异步构建传输队列 - override
    async doBuildQueue(addQueue) {
        throw new Error('function doBuildQueue must override, excute addQueue method');
    };
    // 构建item对象
    doBuildItem(item) { };
    // 获取传输实例对象 - override
    doGetTransferObject(item) {
        // return transferinstance
        throw new Error('function doGetTransferObject must override, return a transfer handler object');
    };
    // 全部传输完成
    onTransferEnd(datas) {
        throw new Error('function onTransferEnd must override');
    };
}

// 传输进度窗体
class TransferFrame extends KFrameAbstract {
    constructor(type) {
        super(type + "-" + Utils.random(), {
            show: false,
            frame: false,
            width: 560,
            height: 300,
            resizable: false,
            webPreferences: {
                nodeIntegration: true
            }
        });
        this._init();
    }
    // 注册事件函数
    IPCHandles(register) {
        // 主窗体: 显示窗口
        register('show', () => { this.show(); });
        // 主窗体: 最小化
        register('mini', () => { this.mini(); });
        // 主窗体: 隐藏
        register('hide', () => { this.hide(); });
        // 主窗体: 关闭
        register('close', () => { this.win.destroy(true); });
        // 主窗体: 最大化 <-> 最小化
        register('togglemax', () => { this.togglemax(); });
    };
    // init
    _init() {
        this.win.on('close', (e) => {
            e.preventDefault();
        });
    };
    // 加载一个实现了传输进度的页面
    async doLoadFile(localfile) {
        return new Promise((resolve, reject) => {
            try {
                this.win.loadFile(localfile);
                this.win.on('ready-to-show', () => {
                    this.win.webContents.send('set-frameid', this.id);
                    resolve(this.win);
                });
            } catch (err) {
                reject(err);
            }
        });
    };
    // 加载一个实现了传输进度的页面
    async doLoadURL(url) {
        return new Promise((resolve, reject) => {
            try {
                this.win.loadURL(url);
                this.win.on('ready-to-show', () => {
                    this.win.webContents.send('set-frameid', this.id);
                    resolve(this.win);
                });
            } catch (err) {
                reject(err);
            }
        });
    };
};
// 传输进度窗体 - 错误
class TransferErrFrame extends KFrameAbstract {
    constructor(parentid) {
        super(parentid + '-err', {
            show: false,
            frame: false,
            width: 560,
            height: 300,
            resizable: false,
            webPreferences: {
                nodeIntegration: true
            },
            parent: FrameTool.get(parentid)
        });
        this.parentid = parentid;
        this._init();
    }
    // 注册事件函数
    IPCHandles(register) {
        // 主窗体: 显示窗口
        register('show', () => { this.show(); });
        // 主窗体: 最小化
        register('mini', () => { this.mini(); });
        // 主窗体: 隐藏
        register('hide', () => { this.hide(); });
        // 主窗体: 关闭
        register('close', () => { this.win.destroy(true); });
        // 主窗体: 最大化 <-> 最小化
        register('togglemax', () => { this.togglemax(); });
    };
    // init
    _init() {
        this.win.on('close', (e) => {
            e.preventDefault();
        });
    };
    // 加载一个实现了传输进度的页面
    async doLoadFile(localfile) {
        return new Promise((resolve, reject) => {
            try {
                this.win.loadFile(localfile);
                this.win.on('ready-to-show', () => {
                    this.win.webContents.send('set-frameid', this.id, this.parentid);
                    resolve(this.win);
                });
            } catch (err) {
                reject(err);
            }
        });
    };
    // 加载一个实现了传输进度的页面
    async doLoadURL(url) {
        return new Promise((resolve, reject) => {
            try {
                this.win.loadURL(url);
                this.win.on('ready-to-show', () => {
                    this.win.webContents.send('set-frameid', this.id, this.parentid);
                    resolve(this.win);
                });
            } catch (err) {
                reject(err);
            }
        });
    };
};