'use strict';
const { KSession, KRequest } = require('ksdk');
/**
 * 文件相关
 */
module.exports = class {
    // 根据路径获取space
    static getSpaceByPath(spacepath) {
        if (spacepath) {
            for (let i = 0; i < KSession.spaceList.length; i++) {
                let space = KSession.spaceList[i]
                if (spacepath.startWith('@' + space.spaceid + '/') || spacepath == ('@' + space.spaceid)) {
                    return space;
                }
            }
        }
        return null;
    };
    // 根据路径获取space 的 APIURL
    static getSpaceAPIURL(spacepath) {
        let res = '';
        if (spacepath) {
            let space = this.getSpaceByPath(spacepath);
            if (space && space.spaceapi) {
                res = space.spaceapi;
            }
        }
        return res;
    };
    // 根据路径获取space 的 spaceroot
    static getSpaceRoot(spacepath) {
        let res = '';
        if (spacepath) {
            let space = this.getSpaceByPath(spacepath);
            if (space && space.spaceroot) {
                res = space.spaceroot;
            }
        }
        return res;
    };
    // 获取node
    static async getnode(path) {
        return new Promise((resolve, reject) => {
            this.getNodes([path]).then((nodes) => {
                resolve(nodes && nodes.length == 1 ? nodes[0] : null);
            }).catch(reject);
        });
    };
    // 获取nodeS
    static async getNodes(paths) {
        return KRequest.postAPI(this.getSpaceAPIURL(paths[0]), { 'op': 'kgetnodes', 'paths': JSON.stringify(paths) });
    };
    // 列出子列表
    static async listdir(dirpath, hasfile, hasdir, depth) {
        return KRequest.postAPI(this.getSpaceAPIURL(dirpath), { 'op': 'klistdir', 'dirpath': dirpath, 'depth': depth, 'hasdir': hasdir, 'hasfile': hasfile });
    };
    // 申请下载地址
    static async askdownload(filepath, datahash, downfilename, downmimetype, expiredsecond, hopesrvs) {
        return KRequest.postAPI(this.getSpaceAPIURL(filepath), {
            'op': 'kaskdownload', 'filepath': filepath, 'expiredsecond': expiredsecond, 'downfilename': downfilename, 'hopesrvs': hopesrvs ? JSON.stringify(hopesrvs) : '',
            'downmimetype': downmimetype, 'datahash': datahash ? datahash.toLowerCase() : ''
        });
    };
    // 创建文件夹
    static async mkdirs(dirpath, metamap) {
        return KRequest.postAPI(this.getSpaceAPIURL(dirpath), { 'op': 'domkdirs', 'dirpath': dirpath, 'metamap': JSON.stringify(metamap) });
    };
    // 删除路径
    static async delete(path) {
        return this.deletes([path]);
    };
    // 删除路径
    static async deletes(paths) {
        return KRequest.postAPI(this.getSpaceAPIURL(paths[0]), { 'op': 'dodeletes', 'paths': JSON.stringify(paths) });
    };
    // 申请上传令牌, 返回 {uptoken, upmethod, httpurl, httpheaders, httpmultiforms, partnumber, partminsize, partmaxsize, datasrv, expiredtime}
    // filemd5_optional/filesize_optional--本地文件MD5及大小(chunksize<0则为可选参数, 否则是必填参数)
    // chunksize--分片大小(-1表示不分片一次性上传)
    // expiredSecond--令牌过期时间
    // hopesrvs--优选的上传数据服务器
    static async askupload(filepath, filemd5, filesize, chunksize, expiredSecond, hopesrvs) {
        return KRequest.postAPI(this.getSpaceAPIURL(filepath), {
            'op': 'kaskupload', 'filepath': filepath, 'filemd5': filemd5 ? filemd5.toLowerCase() : '', 'filesize': filesize,
            'chunksize': chunksize, 'expiredSecond': expiredSecond, 'hopesrvs': hopesrvs ? JSON.stringify(hopesrvs) : ''
        });
    };
    // 兑换 分片上传对象, 返回 {uptoken, upmethod, httpurl, httpheaders, httpmultiforms, partnumber, partminsize, partmaxsize, datasrv, expiredtime}
    static async exchangeupload(filepath, uptoken, partnumber) {
        return KRequest.postAPI(this.getSpaceAPIURL(filepath), { 'op': 'kexchangeupload', 'uptoken': uptoken, 'partnumber': partnumber });
    };
    // 查询 已上传分片，返回 [{uptoken, partnumber(long), partmd5, partsize(long)}]
    static async queryuploaded(filepath, uptoken) {
        return KRequest.postAPI(this.getSpaceAPIURL(filepath), { 'op': 'kqueryuploaded', 'uptoken': uptoken });
    };
    // 递交 上传令牌(若文件已存在则覆盖它)
    static async submitupload(filepath, uptoken, metamap) {
        return KRequest.postAPI(this.getSpaceAPIURL(filepath), { 'op': 'ksubmitupload', 'filepath': filepath, 'uptoken': uptoken, 'meta': metamap });
    };
    // 复制
    static async doputfiles(apiserver, nodes) {
        return KRequest.postAPI(apiserver, { 'op': 'doputfiles', 'filenodes': JSON.stringify(nodes) });
    };
    // 移动
    static async moves(apiserver, srcpaths, destpaths, metamap) {
        return KRequest.postAPI(apiserver, { 'op': 'domoves', 'srcpaths': JSON.stringify(srcpaths), 'destpaths': JSON.stringify(destpaths), 'meta': metamap });
    };
    // 锁定文件
    static async lockfile(filepath) {
        return KRequest.postAPI(this.getSpaceAPIURL(filepath), { 'op': 'lockfile', 'filepath': filepath });
    };
    // 解锁文件
    static async unlockfile(filepath) {
        return KRequest.postAPI(this.getSpaceAPIURL(filepath), { 'op': 'unlockfile', 'filepath': filepath });
    };
};